﻿// Класс для решения диофантовых уравнений
type
  DiophantineSolver = static class
    
    // Расширенный алгоритм Евклида. Возвращает НОД(a,b) и коэффициенты x, y
    public static function ExtendedGcd(a, b: integer; var x, y: integer): integer;
    begin
      if a = 0 then
      begin
        x := 0; y := 1;
        Result := b;
      end
      else
      begin
        var x1, y1: integer;
        var d := ExtendedGcd(b mod a, a, x1, y1);
        x := y1 - (b div a) * x1;
        y := x1;
        Result := d;
      end;
    end;

    // Поиск любого решения уравнения a*x + b*y = c
    public static function FindAnySolution(a, b, c: integer): (integer, integer, boolean);
    begin
      var x0, y0, g: integer;
      g := ExtendedGcd(abs(a), abs(b), x0, y0);
      
      // Решение существует, только если c делится на НОД(a,b)[citation:1][citation:4]
      if c mod g <> 0 then
        Result := (0, 0, false) // решения нет
      else
      begin
        x0 *= c div g;
        y0 *= c div g;
        if a < 0 then x0 := -x0;
        if b < 0 then y0 := -y0;
        Result := (x0, y0, true);
      end;
    end;
    
    
    // Поиск всех решений на заданном отрезке
    public static function FindSolutionsInRange(a, b, c: integer; minX, maxX, minY, maxY: integer): sequence of (integer, integer);
    begin
      var (x0, y0, hasSolution) := FindAnySolution(a, b, c);
      if not hasSolution then exit;
      
      var g := ExtendedGcd(abs(a), abs(b), x0, y0);
      a := a div g; b := b div g;
      
      // Общая формула всех решений[citation:4]:
      // x = x0 + k*(b/g), y = y0 - k*(a/g), где k — любое целое
      // Находим подходящие k для ограничений по x и y
      var kMin := Ceil((minX - x0) / b);
      var kMax := Floor((maxX - x0) / b);
      if b < 0 then Swap(kMin, kMax);
      
      for var k := kMin to kMax do
      begin
        var x := x0 + b * k;
        var y := y0 - a * k;
        if (y >= minY) and (y <= maxY) then
          yield (x, y);
      end;
    end;
  end;
  
  
begin
    // Пример: 3*x + 4*y = 5
  var a := 3; var b := 4; var c := 5;
  
  // 1. Найти любое решение
  var (x, y, exists) := DiophantineSolver.FindAnySolution(a, b, c);
  if exists then
    Writeln($'Одно из решений: x={x}, y={y}')
  else
    Writeln('Решений нет');
  
  // 2. Найти все решения в диапазоне, например, от -10 до 10
  Writeln('Решения в диапазоне x, y ∈ [-10, 10]:');
  foreach var solution in DiophantineSolver.FindSolutionsInRange(a, b, c, -10, 10, -10, 10) do
    Writeln($'  x={solution[0]}, y={solution[1]}');
end.