﻿// ================================
// DiophantineEquations.pas
// Модуль для решения диофантовых уравнений
// ================================

unit DiophantineEquations;

interface

uses System;

type
  /// <summary>Решатель диофантовых уравнений</summary>
  DiophantineSolver = static class
    
    /// <summary>Находит решение уравнения a*x + b*y = c</summary>
    /// <returns>Кортеж (x, y, существует_ли_решение)</returns>
    public static function SolveLinear(a, b, c: integer): (integer, integer, boolean);
    
    /// <summary>Находит все решения в заданном диапазоне</summary>
    /// <returns>Последовательность пар (x, y)</returns>
    public static function FindAllSolutions(a, b, c: integer; 
      minX, maxX, minY, maxY: integer): sequence of (integer, integer);
    
    /// <summary>Проверяет, имеет ли уравнение решение</summary>
    public static function HasSolution(a, b, c: integer): boolean;
  end;

implementation

// Вспомогательная функция: расширенный алгоритм Евклида
function ExtendedGcd(a, b: integer; var x, y: integer): integer;
begin
  if a = 0 then
  begin
    x := 0; y := 1;
    Result := b;
  end
  else
  begin
    var x1, y1: integer;
    var d := ExtendedGcd(b mod a, a, x1, y1);
    x := y1 - (b div a) * x1;
    y := x1;
    Result := d;
  end;
end;

// Реализация методов DiophantineSolver
static function DiophantineSolver.SolveLinear(a, b, c: integer): (integer, integer, boolean);
begin
  var x0, y0, g: integer;
  g := ExtendedGcd(abs(a), abs(b), x0, y0);
  
  if c mod g <> 0 then
    Result := (0, 0, false)
  else
  begin
    x0 := x0 * (c div g);
    y0 := y0 * (c div g);
    
    if a < 0 then x0 := -x0;
    if b < 0 then y0 := -y0;
    
    Result := (x0, y0, true);
  end;
end;

static function DiophantineSolver.HasSolution(a, b, c: integer): boolean;
begin
  var (x, y, exists) := SolveLinear(a, b, c);
  Result := exists;
end;

static function DiophantineSolver.FindAllSolutions(a, b, c: integer; 
  minX, maxX, minY, maxY: integer): sequence of (integer, integer);
begin
  var (x0, y0, hasSolution) := SolveLinear(a, b, c);
  if not hasSolution then exit;
  
  var xTmp, yTmp: integer;
  var g := ExtendedGcd(abs(a), abs(b), xTmp, yTmp);
  
  // Приводим коэффициенты
  var a1 := a div g;
  var b1 := b div g;
  
  // x = x0 + k*b1, y = y0 - k*a1
  // Находим границы для k
  var kMin := Ceil((minX - x0) / b1);
  var kMax := Floor((maxX - x0) / b1);
  
  // Если b1 отрицательный, меняем границы
  if b1 < 0 then
  begin
    var temp := kMin;
    kMin := kMax;
    kMax := temp;
  end;
  
  // Перебираем возможные k
  for var k := kMin to kMax do
  begin
    var x := x0 + b1 * k;
    var y := y0 - a1 * k;
    
    if (y >= minY) and (y <= maxY) then
      yield (x, y);
  end;
end;

end.