﻿// MathExtensions.pas
unit MathExtensions;

{
interface

uses System;

type
  IntegerMathExtensions = static class
  public
    // ========== АРИФМЕТИКА ==========
    /// <summary>Целое число в целую степень (включая отрицательную)</summary>
    //class function Power(self: integer; exponent: integer): real;
    
    /// <summary>Факториал (n!)</summary>
    class function Factorial(self: integer): int64;
    
    /// <summary>Наибольший общий делитель</summary>
    class function GCD(self: integer; b: integer): integer; 
    
    /// <summary>Наименьшее общее кратное</summary>
    //class function LCM(self: integer; b: integer): integer; 
    
    // ========== ПРОВЕРКИ ==========
    /// <summary>Проверка на простоту (оптимизированный алгоритм)</summary>
    class function IsPrime(self: integer): boolean; 
    
    /// <summary>Проверка на совершенное число</summary>
    class function IsPerfect(self: integer): boolean; 
    
    /// <summary>Проверка на палиндром (например, 121)</summary>
    class function IsPalindrome(self: integer): boolean; 
    
    /// <summary>Четность числа</summary>
    class function IsEven(self: integer): boolean; 
    
    /// <summary>Нечетность числа</summary>
    class function IsOdd(self: integer): boolean; 
    
    // ========== РЯДЫ И ПОСЛЕДОВАТЕЛЬНОСТИ ==========
    /// <summary>Числа Фибоначчи до данного индекса</summary>
    class function Fibonacci(self: integer): sequence of integer;
    
    /// <summary>Простые множители числа</summary>
    class function PrimeFactors(self: integer): sequence of integer; 
    
    /// <summary>Делители числа</summary>
    class function Divisors(self: integer): sequence of integer; 
    
    // ========== ПРЕОБРАЗОВАНИЯ ==========
    /// <summary>Двоичное представление</summary>
    class function ToBinary(self: integer): string; 
    
    /// <summary>Обратное число (1/n)</summary>
    class function Reciprocal(self: integer): real; 
    
    /// <summary>Римская запись числа (до 3999)</summary>
    class function ToRoman(self: integer): string; 
    
    // ========== ДИАПАЗОНЫ ==========
    /// <summary>Последовательность от 1 до n</summary>
    class function UpTo(self: integer): sequence of integer; 
    
    /// <summary>Проверка принадлежности диапазону [a, b]</summary>
    class function InRange(self: integer; a, b: integer): boolean;
  end;

  RealMathExtensions = static class
  public
    /// <summary>Округление до N знаков после запятой</summary>
    class function RoundTo(self: real; decimals: integer): real;
    
    /// <summary>Проверка на целое число</summary>
    class function IsInteger(self: real): boolean; 
    
    /// <summary>Процент от числа</summary>
    class function PercentOf(self: real; percent: real): real;
  end;

implementation

// ========== ВСПОМОГАТЕЛЬНЫЕ ФУНКЦИИ ==========
function FastPower(base: integer; exp: integer): real;
begin
  if exp = 0 then Result := 1
  else if exp mod 2 = 0 then 
    Result := FastPower(base * base, exp div 2)
  else 
    Result := base * FastPower(base, exp - 1);
end;

// ========== РЕАЛИЗАЦИЯ ДЛЯ INTEGER ==========
class function IntegerMathExtensions.Power(self: integer; exponent: integer): real;
begin
  if (self = 0) and (exponent <= 0) then
    raise new System.ArgumentException('Недопустимая операция 0^' + exponent);
  
  if exponent = 0 then Result := 1
  else if exponent > 0 then Result := FastPower(self, exponent)
  else Result := 1 / FastPower(self, -exponent);
end;

class function IntegerMathExtensions.Factorial(self: integer): int64;
begin
  if self < 0 then 
    raise new System.ArgumentException('Факториал определен для n >= 0');
  
  Result := 1;
  for var i := 2 to self do
    Result *= i;
end;

class function IntegerMathExtensions.GCD(self: integer; b: integer): integer;
begin
  var a := Abs(self);
  b := Abs(b);
  while b <> 0 do
  begin
    var temp := b;
    b := a mod b;
    a := temp;
  end;
  Result := a;
end;

{
class function IntegerMathExtensions.LCM(self: integer; b: integer): integer;
begin
  if (self = 0) or (b = 0) then 
    Result := 0
  else 
    Result := Abs(self * b) div self.GCD(b);
end;
}
class function IntegerMathExtensions.IsPrime(self: integer): boolean;
begin
  if self <= 1 then Exit(false);
  if self <= 3 then Exit(true);
  if (self mod 2 = 0) or (self mod 3 = 0) then Exit(false);
  
  var i := 5;
  while i * i <= self do
  begin
    if (self mod i = 0) or (self mod (i + 2) = 0) then 
      Exit(false);
    i += 6;
  end;
  Result := true;
end;

class function IntegerMathExtensions.IsPerfect(self: integer): boolean;
begin
  if self <= 1 then Exit(false);
  
  var sum := 1;
  for var i := 2 to Round(Sqrt(self)) do
    if self mod i = 0 then
    begin
      sum += i;
      if i <> self div i then 
        sum += self div i;
    end;
  
  Result := sum = self;
end;

class function IntegerMathExtensions.IsPalindrome(self: integer): boolean;
begin
  var s := self.ToString;
  for var i := 1 to s.Length div 2 do
    if s[i] <> s[s.Length - i + 1] then 
      Exit(false);
  Result := true;
end;

class function IntegerMathExtensions.IsEven(self: integer): boolean;
begin
  Result := self mod 2 = 0;
end;

class function IntegerMathExtensions.IsOdd(self: integer): boolean;
begin
  Result := not self.IsEven;
end;

class function IntegerMathExtensions.Fibonacci(self: integer): sequence of integer;
begin
  //if self <= 0 then yield break;
  
  var (a, b) := (0, 1);
  for var i := 1 to self do
  begin
    yield a;
    (a, b) := (b, a + b);
  end;
end;

class function IntegerMathExtensions.PrimeFactors(self: integer): sequence of integer;
begin
  var n := self;
  var divisor := 2;
  
  while n > 1 do
  begin
    while n mod divisor = 0 do
    begin
      yield divisor;
      n := n div divisor;
    end;
    
    if divisor = 2 then divisor += 1
    else divisor += 2;
    
    if divisor * divisor > n then
    begin
      if n > 1 then yield n;
      break;
    end;
  end;
end;

class function IntegerMathExtensions.Divisors(self: integer): sequence of integer;
begin
  for var i := 1 to Round(Sqrt(self)) do
    if self mod i = 0 then
    begin
      yield i;
      if i <> self div i then 
        yield self div i;
    end;
end;

class function IntegerMathExtensions.ToBinary(self: integer): string;
begin
  Result := '';
  var n := self;
  repeat
    Result := (n mod 2).ToString + Result;
    n := n div 2;
  until n = 0;
end;

class function IntegerMathExtensions.Reciprocal(self: integer): real;
begin
  if self = 0 then
    raise new System.DivideByZeroException('Деление на ноль');
  Result := 1.0 / self;
end;


  const roman: array of string = 
    ('M','CM','D','CD','C','XC','L','XL','X','IX','V','IV','I');
  const values: array of integer = 
    (1000, 900, 500, 400, 100, 90, 50, 40, 10, 9, 5, 4, 1);
class function IntegerMathExtensions.ToRoman(self: integer): string;
begin
  if (self < 1) or (self > 3999) then
    raise new System.ArgumentOutOfRangeException('Римские цифры: 1..3999');
  var n := self;
  Result := '';
  
  for var i := 0 to roman.High do
    while n >= values[i] do
    begin
      Result += roman[i];
      n -= values[i];
    end;
end;

class function IntegerMathExtensions.UpTo(self: integer): sequence of integer;
begin
  for var i := 1 to self do
    yield i;
end;

class function IntegerMathExtensions.InRange(self: integer; a, b: integer): boolean;
begin
  Result := (self >= a) and (self <= b);
end;

// ========== РЕАЛИЗАЦИЯ ДЛЯ REAL ==========
class function RealMathExtensions.RoundTo(self: real; decimals: integer): real;
begin
  var factor := Power(10, decimals);
  Result := Round(self * factor) / factor;
end;

class function RealMathExtensions.IsInteger(self: real): boolean;
begin
  Result := Abs(Frac(self)) < 1e-10;
end;


class function RealMathExtensions.PercentOf(self: real; percent: real): real;
begin
  Result := self * percent / 100;
end;

end.

