﻿unit FibonacciGenerator;

interface

type
  Fibonacci = static class
  public
    /// Стандартная последовательность (0, 1, 1, 2, 3...)
    static function Classic: sequence of int64;
    
    /// Альтернативная последовательность (1, 2, 3, 5, 8...)
    static function Alternative: sequence of int64;
    
    /// Последовательность с заданными начальными значениями
    static function Custom(first, second: int64): sequence of int64;
    
    /// Получить N первых чисел
    static function Take(count: integer; first: int64 := 0; second: int64 := 1): array of int64;
    
    /// Получить N-ное число Фибоначчи
    static function GetNth(n: integer; first: int64 := 0; second: int64 := 1): int64;
    
    /// Проверить, является ли число числом Фибоначчи
    static function IsFibonacci(num: int64): boolean;
  end;

implementation

class function Fibonacci.Classic: sequence of int64;
begin
  var a: int64 := 0;
  var b: int64 := 1;
  
  while true do
  begin
    yield a;
    var next := a + b;
    a := b;
    b := next;
  end;
end;

class function Fibonacci.Alternative: sequence of int64;
begin
  var a: int64 := 1;
  var b: int64 := 2;
  
  while true do begin
    yield a;
    (a, b) := (b, (a + b));
  end;
end;

class function Fibonacci.Custom(first, second: int64): sequence of int64;
begin
  var a := first;
  var b := second;
  
  while true do
  begin
    yield a;
    (a, b) := (b, (a + b));
  end;
end;

class function Fibonacci.Take(count: integer; first, second: int64): array of int64;
begin
  Result := new int64[count];
  var a := first;
  var b := second;
  
  for var i := 0 to count - 1 do
  begin
    Result[i] := a;
    (a, b) := (b, (a + b));
  end;
end;

class function Fibonacci.GetNth(n: integer; first, second: int64): int64;
begin
  if n < 0 then
    raise new System.ArgumentOutOfRangeException('n', 'n должно быть неотрицательным');
    
  if n = 0 then
    Exit(first);
    
  var a := first;
  var b := second;
  
  for var i := 2 to n do
  begin
    var next := a + b;
    a := b;
    b := next;
  end;
  
  Result := b;
end;


  function IsPerfectSquare(x: int64): boolean;
  begin
    var sqrtX := Round(Sqrt(x));
    Result := sqrtX * sqrtX = x;
  end;
class function Fibonacci.IsFibonacci(num: int64): boolean;
begin
  // Число является числом Фибоначчи, если
  // одно из выражений 5*n^2 + 4 или 5*n^2 - 4 является полным квадратом
  
  var check1 := 5 * num * num + 4;
  var check2 := 5 * num * num - 4;
  Result := IsPerfectSquare(check1) or IsPerfectSquare(check2);
end;

end.

{
// Основная программа
begin
  Writeln('Генератор чисел Фибоначчи');
  Writeln('=========================');
  
  Writeln(#10'Стандартная последовательность (первые 20):');
  foreach var n in Fibonacci.Standard.Take(20) do
    Write(n, ' ');
  
  Writeln(#10#10'Альтернативная последовательность (первые 20):');
  foreach var n in Fibonacci.Alternative.Take(20) do
    Write(n, ' ');
  
  Writeln(#10#10'Кастомная последовательность (3, 5, 8...):');
  foreach var n in Fibonacci.Custom(3, 5).Take(15) do
    Write(n, ' ');
  
  Writeln(#10#10'Получить первые 10 чисел как массив:');
  var arr := Fibonacci.Take(10);
  Writeln(string.Join(', ', arr));
  
  Writeln(#10'Некоторые конкретные числа:');
  Writeln('F[10] = ', Fibonacci.GetNth(10));
  Writeln('F[20] = ', Fibonacci.GetNth(20));
  Writeln('F[30] = ', Fibonacci.GetNth(30));
  
  Writeln(#10'Проверка чисел на принадлежность последовательности:');
  var testNumbers := Arr(0, 1, 2, 3, 4, 5, 8, 13, 20, 21, 34, 55);
  foreach var n in testNumbers do
    Writeln(n, ' - ', Fibonacci.IsFibonacci(n));
end.
}