﻿unit PythagoreanTriples;

interface

type
  // Используем другое имя, чтобы избежать конфликта
  PythagTriple = record
    a, b, c: integer;
    
    constructor(na, nb, nc: integer);
    begin
      a := na;
      b := nb;
      c := nc;
    end;
    
    function IsPrimitive: boolean;
    begin
      Result := GCD(a, b) = 1;
    end;
    
    function Perimeter: integer := a + b + c;
    function Area: integer := (a * b) div 2;
    
    function ToString: string; override;
    begin
      Result := $'({a}, {b}, {c})';
    end;
  end;

  PythagoreanGenerator = static class
  public
    // Генерация примитивных троек (взаимно простых)
    class function GeneratePrimitive(maxC: integer): List<PythagTriple>;
    
    // Генерация всех троек (включая кратные)
    class function GenerateAll(maxC: integer): List<PythagTriple>;
    
    // Генерация троек с заданным периметром
    class function WithPerimeter(P: integer): List<PythagTriple>;
    
    // Генерация троек с заданной гипотенузой
    class function WithHypotenuse(c: integer): List<PythagTriple>;
    
    // Проверка, является ли тройка пифагоровой
    class function IsPythagorean(a, b, c: integer): boolean;
    
    // Получение всех троек, содержащих заданное число
    class function ContainingNumber(n: integer; maxC: integer): List<PythagTriple>;
  end;

implementation

// Вспомогательная функция для НОД
function GCD(x, y: integer): integer;
begin
  while y <> 0 do
  begin
    var temp := y;
    y := x mod y;
    x := temp;
  end;
  Result := x;
end;

class function PythagoreanGenerator.GeneratePrimitive(maxC: integer): List<PythagTriple>;
begin
  Result := new List<PythagTriple>;
  
  for var m := 2 to Trunc(Sqrt(maxC)) + 1 do
    for var n := 1 to m - 1 do
    begin
      if ((m - n) mod 2 = 1) and (GCD(m, n) = 1) then
      begin
        var a := m * m - n * n;
        var b := 2 * m * n;
        var c := m * m + n * n;
        
        if c <= maxC then
        begin
          // Упорядочиваем
          if a > b then Swap(a, b);
          Result.Add(new PythagTriple(a, b, c));
        end;
      end;
    end;
  
  Result.Sort((t1, t2) -> t1.c.CompareTo(t2.c));
end;

class function PythagoreanGenerator.GenerateAll(maxC: integer): List<PythagTriple>;
begin
  Result := new List<PythagTriple>;
  var primitive := GeneratePrimitive(maxC);
  
  // Добавляем примитивные
  Result.AddRange(primitive);
  
  // Добавляем кратные
  for var k := 2 to maxC div 5 do
  begin
    foreach var triple in primitive do
    begin
      var newTriple := new PythagTriple(triple.a * k, triple.b * k, triple.c * k);
      if newTriple.c <= maxC then
        Result.Add(newTriple);
    end;
  end;
  
  Result.Sort((t1, t2) -> begin
    var cmp := t1.c.CompareTo(t2.c);
    if cmp = 0 then cmp := t1.a.CompareTo(t2.a);
    Result := cmp;
  end);
end;

class function PythagoreanGenerator.WithPerimeter(P: integer): List<PythagTriple>;
begin
  Result := new List<PythagTriple>;
  
  for var a := 1 to P div 3 do
    for var b := a + 1 to (P - a) div 2 do
    begin
      var c := P - a - b;
      if (c > b) and (a * a + b * b = c * c) then
        Result.Add(new PythagTriple(a, b, c));
    end;
end;

class function PythagoreanGenerator.WithHypotenuse(c: integer): List<PythagTriple>;
begin
  Result := new List<PythagTriple>;
  
  for var a := 1 to Trunc(c / Sqrt(2)) do
  begin
    var bSquared := c * c - a * a;
    var b := Trunc(Sqrt(bSquared));
    
    if (b * b = bSquared) and (a < b) then
      Result.Add(new PythagTriple(a, b, c));
  end;
end;

class function PythagoreanGenerator.IsPythagorean(a, b, c: integer): boolean;
begin
  var nums := |a, b, c|;
  nums.Sort;
  Result := nums[0] * nums[0] + nums[1] * nums[1] = nums[2] * nums[2];
end;

class function PythagoreanGenerator.ContainingNumber(n: integer; maxC: integer): List<PythagTriple>;
begin
  Result := new List<PythagTriple>;
  var allTriples := GenerateAll(maxC);
  
  foreach var triple in allTriples do
    if (triple.a = n) or (triple.b = n) or (triple.c = n) then
      Result.Add(triple);
end;

end.